 /******************************************************************************
 *
 *	   MIKADO LEVEL    M M A 8 4 5 1 Q    A C C E L E R O M E T E R   C O D E 
 *
 *	   Jan 2012 - Geoff Champion - Base s/w - Silicon Chip Inclinometer and G Force Meter
 *
 *     April 2020 - Upgraded to XC8 compiler
 *
 */
#include "accel.h"
#include "display.h"
#include <math.h>
#include <xc.h>

#define CAL_ADDRESS 		0x00
#define FLAG_ADDRESS		0x04
#define MODE_ADDRESS        0x08
#define MAX_ADDRESS         0x0C

unsigned char rawdata[6];
float theta, phi, rho;
float thetaraw, thetaoffset;

signed int angle;

unsigned char PSMode;
uchar max_angle = 0;

//#####################################################################################

// Configure Acceleromemeter	

void accInitialise(void) 
{
    accSet(F_SETUP, 0);             		// FIFO disabled
    accSet(XYZ_DATA_CFG, 0);        		// 2g, HPF disabled
    accSet(CTRL_REG1, 0xb0);        		// 2-byte read, 6.25Hz awake & asleep
    accSet(CTRL_REG2, 0x18);        		// Normal res awake & asleep, no auto sleep
    accSet(CTRL_REG3, 0x00);        		// Int PP, active lo, no wake-up
    accSet(CTRL_REG4, 0x01);        		// Data rdy,  int enabled
    accSet(CTRL_REG5, 0x01);        		// Data rdy int ->int1, rest -> int2
}

//#####################################################################################

void accStart(void) 
{
    uchar temp;
    
    // Put into active mode
    i2cSetByte(ACC_ADDR, CTRL_REG1, i2cGetByte(ACC_ADDR, CTRL_REG1) | 0x01);
    
    // get calibration and mode value
    thetaoffset = 0.0;
    if (eeReadChar(FLAG_ADDRESS) == 0x00) {  // Good values
        thetaoffset = eeReadFloat(CAL_ADDRESS);
    }
    else {
        eeWriteFloat(CAL_ADDRESS, 0.00);
        eeWriteChar(FLAG_ADDRESS, 0x00);
    }
	
	PSMode = eeReadChar(MODE_ADDRESS);
    
    // Enable interrupts
    INTCON3bits.INT1IF = 0;                 // Clear int1 flag
    INTCON3bits.INT1IE = 1;                 // enable int1
    INTCON3bits.INT2IF = 0;                 // Clear int2 flag
    INTCON3bits.INT2IE = 1;                 // enable int2
    temp = CAL_PIN;                         // clear any mismatch and
    INTCONbits.RABIF = 0;                   // Clear IOC flag
    INTCONbits.RABIE = 1;                   // Enable interrupt on change
}

//#####################################################################################

void accSet(uchar reg, uchar value) 
{
    i2cSetByte(ACC_ADDR, reg, value);
}

//#####################################################################################

//uchar accGet(uchar reg) 
//{
//    return i2cGetByte(ACC_ADDR, reg);
//}

//#####################################################################################

void accInt1isr(void) 
{
    int accx, accy, accz;
long cnt;
    i2cRead(ACC_ADDR, OUTPUT_REGS, 6, rawdata);
    accx = ((unsigned int) rawdata[0] << 8) + rawdata[1];
    accy = ((unsigned int) rawdata[2] << 8) + rawdata[3];
    accz = ((unsigned int) rawdata[4] << 8) + rawdata[5];
    theta = atan2(-accx, accy) * 57.2957795 + 180;
    thetaraw = theta;
    theta += thetaoffset;

    if (theta >= 180.0) theta -= 360.0;

    phi = fabs(atan2(accy, -accz)) * 57.2957795;
    rho = fabs(atan2(accz, accx)) * 57.2957795;
    
    angle = (unsigned int) theta;
    
    if ((phi >= 45.0 && phi <= 135.0) || (rho <= 45.0 || rho >= 135.0)) 
    {
		max_angle = (unsigned int) eeReadFloat(MAX_ADDRESS);

		if (fabs(angle) <= 180)
		{
	    	if (fabs(angle) > max_angle) eeWriteFloat(MAX_ADDRESS, fabs(angle));
  		}  	
    	dispHeel(angle);
	}
	else dispBlank();
}

//#####################################################################################

void accCalisr() 
{
    unsigned char temp;

    if (CAL_PIN == 0) {                     // Pressed
        TMR0H = 0;                          // start timer
        TMR0L = 0;
        T0CONbits.TMR0ON = 1;
        INTCONbits.TMR0IE = 1;
        INTCONbits.TMR0IF = 0;
    } else {                                // Released
        temp = TMR0L;
        temp = TMR0H;
        T0CONbits.TMR0ON = 0;               // stop timer
        INTCONbits.TMR0IE = 0;
        if (temp < 2) return;               // less that 30 ms -> ignore
        if (temp < 122) {                   // less than 2sec -> cal
            if (thetaraw < 45.0) thetaoffset = -thetaraw;
            else if (thetaraw > 315.0) thetaoffset = 360.0 - thetaraw;
            eeWriteFloat(CAL_ADDRESS, thetaoffset);
        }
    }
    temp = CAL_PIN;
    INTCONbits.RABIF = 0;
}

//#####################################################################################

void accTimeout(void)
{
	PSMode = eeReadChar(MODE_ADDRESS);
	
	PSMode = (PSMode == 0 ? 1 : 0);
	eeWriteChar(MODE_ADDRESS, PSMode);
	
	T0CONbits.TMR0ON = 0;
	INTCONbits.TMR0IE = 0;
	dispDot();
}
